<?php
// app/controllers/QuoteController.php (v2.1 extras + edición)
declare(strict_types=1);

class QuoteController {
    public function __construct(private PDO $db, private array $config) {}

    private function hasExtras(): bool {
        $stmt = $this->db->query("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'quotes' AND COLUMN_NAME IN ('installation_desc','installation_cost','supplies_desc','supplies_cost')");
        $cols = array_column($stmt->fetchAll(), 'COLUMN_NAME');
        return in_array('installation_cost', $cols, true) || in_array('supplies_cost', $cols, true);
    }

    private function normalizeItems(array $items): array {
        $normalized = [];
        if (isset($items[0]) && is_array($items[0])) {
            foreach ($items as $row) {
                $pid = (int)($row['product_id'] ?? 0);
                $qty = (int)($row['qty'] ?? 1);
                $normalized[] = ['product_id' => $pid, 'qty' => $qty];
            }
        } elseif (isset($items['product_id']) && is_array($items['product_id'])) {
            $pids = $items['product_id'];
            $qtys = $items['qty'] ?? [];
            $count = max(count($pids), count($qtys));
            for ($i=0; $i<$count; $i++) {
                $pid = (int)($pids[$i] ?? 0);
                $qty = (int)($qtys[$i] ?? 1);
                $normalized[] = ['product_id' => $pid, 'qty' => $qty];
            }
        }
        return $normalized;
    }

    public function index(): void {
        $chk = $this->db->query("SELECT COUNT(*) AS n FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'quotes' AND COLUMN_NAME = 'client_id'");
        $hasClientCol = (int)($chk->fetch()['n'] ?? 0) > 0;
        $chkT = $this->db->query("SELECT COUNT(*) AS n FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'clients'");
        $hasClientsTable = (int)($chkT->fetch()['n'] ?? 0) > 0;

        $clientId = (int)($_GET['client_id'] ?? 0);
        $q = trim($_GET['q'] ?? '');

        $select = "SELECT q.*, u.name as user_name";
        $join   = " FROM quotes q JOIN users u ON q.user_id = u.id";
        if ($hasClientCol && $hasClientsTable) {
            $select .= ", c.name as client_name";
            $join   .= " LEFT JOIN clients c ON q.client_id = c.id";
        }

        $sql = $select . $join;
        $where = []; $params = [];
        if ($hasClientCol && $clientId > 0) { $where[] = "q.client_id = ?"; $params[] = $clientId; }
        if ($q !== '') { $where[] = "(q.customer LIKE ? OR q.id = ?)"; $params[] = "%$q%"; $params[] = (int)$q; }
        if ($where) { $sql .= " WHERE " . implode(" AND ", $where); }
        $sql .= " ORDER BY q.id DESC";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $quotes = $stmt->fetchAll();

        $clientsFilter = [];
        if ($hasClientsTable) {
            try {
                $c = $this->db->query("SELECT id, name, company FROM clients ORDER BY name ASC");
                $clientsFilter = $c->fetchAll();
            } catch (Throwable $e) {}
        }

        $title = 'Cotizaciones';
        require __DIR__ . '/../views/partials/header.php';
        require __DIR__ . '/../views/quotes/index.php';
        require __DIR__ . '/../views/partials/footer.php';
    }

    public function create(): void {
        $p = $this->db->query("SELECT id, sku, name, price, stock FROM products ORDER BY name ASC");
        $products = $p->fetchAll();
        $clients = [];
        try { $c = $this->db->query("SELECT id, name, company FROM clients ORDER BY name ASC"); $clients = $c->fetchAll(); } catch (Throwable $e) {}
        $title = 'Nueva cotización';
        require __DIR__ . '/../views/partials/header.php';
        require __DIR__ . '/../views/quotes/create.php';
        require __DIR__ . '/../views/partials/footer.php';
    }

    public function store(): void {
        $clientId = (int)($_POST['client_id'] ?? 0);
        $customer = trim($_POST['customer'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        $installation_desc = trim($_POST['installation_desc'] ?? '');
        $installation_cost = (float)($_POST['installation_cost'] ?? 0);
        $supplies_desc = trim($_POST['supplies_desc'] ?? '');
        $supplies_cost = (float)($_POST['supplies_cost'] ?? 0);

        $itemsRaw = $_POST['items'] ?? [];
        $items = $this->normalizeItems(is_array($itemsRaw) ? $itemsRaw : []);

        $this->db->beginTransaction();
        try {
            if ($clientId > 0) {
                $c = $this->db->prepare("SELECT name, company FROM clients WHERE id=?");
                $c->execute([$clientId]);
                $cRow = $c->fetch();
                if ($cRow) { $customer = $cRow['name'] . (empty($cRow['company']) ? '' : ' — ' . $cRow['company']); }
            }

            $hasExtras = $this->hasExtras();

            if ($hasExtras) {
                $stmt = $this->db->prepare("INSERT INTO quotes (customer, client_id, notes, installation_desc, installation_cost, supplies_desc, supplies_cost, user_id) VALUES (?,?,?,?,?,?,?,?)");
                $stmt->execute([$customer, $clientId ?: null, $notes, $installation_desc, $installation_cost, $supplies_desc, $supplies_cost, $_SESSION['user']['id']]);
            } else {
                $stmt = $this->db->prepare("INSERT INTO quotes (customer, client_id, notes, user_id) VALUES (?,?,?,?)");
                $stmt->execute([$customer, $clientId ?: null, $notes, $_SESSION['user']['id']]);
            }
            $quoteId = (int)$this->db->lastInsertId();

            $qItem = $this->db->prepare("INSERT INTO quote_items (quote_id, product_id, qty, price) VALUES (?,?,?,?)");
            $sum = 0.0;
            foreach ($items as $it) {
                $pid = (int)($it['product_id'] ?? 0);
                $qty = max(1, (int)($it['qty'] ?? 1));
                if ($pid <= 0) continue;
                $p = $this->db->prepare("SELECT price FROM products WHERE id=?");
                $p->execute([$pid]);
                $row = $p->fetch();
                if (!$row) continue;
                $price = (float)$row['price'];
                $qItem->execute([$quoteId, $pid, $qty, $price]);
                $sum += $price * $qty;
            }
            if ($hasExtras) {
                $sum += max(0, (float)$installation_cost) + max(0, (float)$supplies_cost);
            }
            $this->db->prepare("UPDATE quotes SET total = ? WHERE id = ?")->execute([$sum, $quoteId]);
            $this->db->commit();
        } catch (Throwable $e) {
            $this->db->rollBack();
            $_SESSION['error'] = 'Error al guardar la cotización: ' . $e->getMessage();
        }
        header('Location: ?r=quotes'); exit;
    }

    public function edit(): void {
        $id = (int)($_GET['id'] ?? 0);
        $stmt = $this->db->prepare("SELECT * FROM quotes WHERE id=?");
        $stmt->execute([$id]);
        $quote = $stmt->fetch();
        if (!$quote) { http_response_code(404); echo 'Cotización no encontrada'; return; }

        $itemsStmt = $this->db->prepare("SELECT qi.*, p.sku, p.name FROM quote_items qi JOIN products p ON qi.product_id = p.id WHERE qi.quote_id = ?");
        $itemsStmt->execute([$id]);
        $items = $itemsStmt->fetchAll();

        $p = $this->db->query("SELECT id, sku, name, price, stock FROM products ORDER BY name ASC");
        $products = $p->fetchAll();
        $clients = [];
        try { $c = $this->db->query("SELECT id, name, company FROM clients ORDER BY name ASC"); $clients = $c->fetchAll(); } catch (Throwable $e) {}

        $title = 'Editar cotización #' . $id;
        require __DIR__ . '/../views/partials/header.php';
        require __DIR__ . '/../views/quotes/edit.php';
        require __DIR__ . '/../views/partials/footer.php';
    }

    public function update(): void {
        $id = (int)($_POST['id'] ?? 0);
        $clientId = (int)($_POST['client_id'] ?? 0);
        $customer = trim($_POST['customer'] ?? '');
               $notes = trim($_POST['notes'] ?? '');
        $installation_desc = trim($_POST['installation_desc'] ?? '');
        $installation_cost = (float)($_POST['installation_cost'] ?? 0);
        $supplies_desc = trim($_POST['supplies_desc'] ?? '');
        $supplies_cost = (float)($_POST['supplies_cost'] ?? 0);

        $itemsRaw = $_POST['items'] ?? [];
        $items = $this->normalizeItems(is_array($itemsRaw) ? $itemsRaw : []);

        $this->db->beginTransaction();
        try {
            if ($clientId > 0) {
                $c = $this->db->prepare("SELECT name, company FROM clients WHERE id=?");
                $c->execute([$clientId]);
                $cRow = $c->fetch();
                if ($cRow) { $customer = $cRow['name'] . (empty($cRow['company']) ? '' : ' — ' . $cRow['company']); }
            }
            $hasExtras = $this->hasExtras();
            if ($hasExtras) {
                $u = $this->db->prepare("UPDATE quotes SET customer=?, client_id=?, notes=?, installation_desc=?, installation_cost=?, supplies_desc=?, supplies_cost=? WHERE id=?");
                $u->execute([$customer, $clientId ?: null, $notes, $installation_desc, $installation_cost, $supplies_desc, $supplies_cost, $id]);
            } else {
                $u = $this->db->prepare("UPDATE quotes SET customer=?, client_id=?, notes=? WHERE id=?");
                $u->execute([$customer, $clientId ?: null, $notes, $id]);
            }

            $this->db->prepare("DELETE FROM quote_items WHERE quote_id=?")->execute([$id]);
            $qItem = $this->db->prepare("INSERT INTO quote_items (quote_id, product_id, qty, price) VALUES (?,?,?,?)");

            $sum = 0.0;
            foreach ($items as $it) {
                $pid = (int)($it['product_id'] ?? 0);
                $qty = max(1, (int)($it['qty'] ?? 1));
                if ($pid <= 0) continue;
                $p = $this->db->prepare("SELECT price FROM products WHERE id=?");
                $p->execute([$pid]);
                $row = $p->fetch();
                if (!$row) continue;
                $price = (float)$row['price'];
                $qItem->execute([$id, $pid, $qty, $price]);
                $sum += $price * $qty;
            }
            if ($hasExtras) {
                $sum += max(0, (float)($installation_cost)) + max(0, (float)($supplies_cost));
            }
            $this->db->prepare("UPDATE quotes SET total = ? WHERE id = ?")->execute([$sum, $id]);

            $this->db->commit();
        } catch (Throwable $e) {
            $this->db->rollBack();
            $_SESSION['error'] = 'Error al actualizar la cotización: ' . $e->getMessage();
        }

        header('Location: ?r=quotes_view&id=' . $id); exit;
    }

    public function view(): void {
        $id = (int)($_GET['id'] ?? 0);
        $stmt = $this->db->prepare("SELECT q.*, u.name as user_name, c.name as client_name, c.company, c.email, c.phone, c.address 
                                    FROM quotes q JOIN users u ON q.user_id = u.id 
                                    LEFT JOIN clients c ON q.client_id = c.id WHERE q.id=?");
        try { $stmt->execute([$id]); }
        catch (Throwable $e) { $stmt = $this->db->prepare("SELECT q.*, u.name as user_name FROM quotes q JOIN users u ON q.user_id = u.id WHERE q.id=?"); $stmt->execute([$id]); }
        $quote = $stmt->fetch();
        if (!$quote) { http_response_code(404); echo 'Cotización no encontrada'; return; }

        $items = $this->db->prepare("SELECT qi.*, p.sku, p.name FROM quote_items qi JOIN products p ON qi.product_id = p.id WHERE qi.quote_id = ?");
        $items->execute([$id]);
        $items = $items->fetchAll();

        $title = 'Cotización #' . $id;
        require __DIR__ . '/../views/partials/header.php';
        require __DIR__ . '/../views/quotes/view.php';
        require __DIR__ . '/../views/partials/footer.php';
    }

    public function printView(): void {
        $id = (int)($_GET['id'] ?? 0);
        $stmt = $this->db->prepare("SELECT q.*, u.name as user_name, c.name as client_name, c.company, c.email, c.phone, c.address 
                                    FROM quotes q JOIN users u ON q.user_id = u.id 
                                    LEFT JOIN clients c ON q.client_id = c.id WHERE q.id=?");
        try { $stmt->execute([$id]); }
        catch (Throwable $e) { $stmt = $this->db->prepare("SELECT q.*, u.name as user_name FROM quotes q JOIN users u ON q.user_id = u.id WHERE q.id=?"); $stmt->execute([$id]); }
        $quote = $stmt->fetch();
        if (!$quote) { http_response_code(404); echo 'Cotización no encontrada'; return; }

        $items = $this->db->prepare("SELECT qi.*, p.sku, p.name FROM quote_items qi JOIN products p ON qi.product_id = p.id WHERE qi.quote_id = ?");
        $items->execute([$id]);
        $items = $items->fetchAll();

        $company = $this->config['company'];
        $title = 'Imprimir Cotización';
        require __DIR__ . '/../views/quotes/print.php';
    }
}